<?php
// database.php
require_once 'config.php';

class Database {
    private $pdo;
    private static $instance = null;

    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->createTables();
        } catch (PDOException $e) {
            error_log("DB Connection Error: " . $e->getMessage(), 3, LOG_FILE);
            die("Database Error");
        }
    }

    public static function getInstance() {
        if (self::$instance == null) {
            self::$instance = new Database();
        }
        return self::$instance->pdo;
    }

    private function createTables() {
              // جدول کاربران
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS users (
            id BIGINT PRIMARY KEY,
            username VARCHAR(255),
            name VARCHAR(255),
            phone VARCHAR(50),
            step VARCHAR(100) DEFAULT 'none',
            temp_data TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        // --- کد تکمیلی برای افزودن ستون last_activity به دیتابیس‌های قدیمی ---
        // این بخش چک می‌کند اگر ستون وجود نداشت، آن را اضافه کند
        try {
            $this->pdo->exec("ALTER TABLE users ADD COLUMN last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP");
        } catch (Exception $e) {
            // اگر ستون از قبل وجود داشته باشد، خطا می‌دهد که ما آن را نادیده می‌گیریم
        }


        // جدول ادمین‌های ربات
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS admins (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chat_id BIGINT UNIQUE,
            name VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");


        // جدول محصولات
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS products (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255),
            support_days INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول محصولات کاربر
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS user_products (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT,
            product_id INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول دپارتمان‌ها
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS departments (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255),
            parent_id INT DEFAULT NULL,
            require_product TINYINT(1) DEFAULT 0,
            FOREIGN KEY (parent_id) REFERENCES departments(id) ON DELETE CASCADE
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول پشتیبان‌های دپارتمان
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS dept_agents (
            id INT AUTO_INCREMENT PRIMARY KEY,
            dept_id INT,
            agent_name VARCHAR(255),
            agent_id BIGINT,
            FOREIGN KEY (dept_id) REFERENCES departments(id) ON DELETE CASCADE
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول تیکت‌ها
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS tickets (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT,
            dept_id INT,
            product_id INT DEFAULT NULL,
            priority VARCHAR(50),
            title VARCHAR(255),
            status VARCHAR(50) DEFAULT 'open', -- open, answered, closed
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (dept_id) REFERENCES departments(id) ON DELETE SET NULL
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول پیام‌های تیکت
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS ticket_messages (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ticket_id INT,
            sender_id BIGINT,
            sender_type VARCHAR(20), -- user, agent
            message_type VARCHAR(20), -- text, photo, voice, document, video
            content TEXT, -- text content or caption
            file_id VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (ticket_id) REFERENCES tickets(id) ON DELETE CASCADE
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
                // --- جداول مربوط به هوش مصنوعی ---

        // جدول کلیدهای API
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS ai_api_keys (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255),
            api_key VARCHAR(255),
            usage_count INT DEFAULT 0,
            fail_count INT DEFAULT 0,
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // جدول مستندات (Knowledge Base)
        // از نوع LONGTEXT استفاده می‌کنیم تا محدودیت حجم متن بسیار بالا باشد
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS ai_docs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255),
            content LONGTEXT,
            file_type VARCHAR(10) DEFAULT 'text', -- text, pdf, docx, txt
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
                // جدول لاگ‌های هوش مصنوعی (برای عیب‌یابی دقیق ارورهای خالی)
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS ai_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT,
            input_text TEXT,
            raw_response LONGTEXT, -- پاسخ خام سرور برای دیدن خطاهای HTML
            error_message TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

                // --- جداول مربوط به تنظیمات ربات (قابلیت جدید) ---

        // جدول تنظیمات عمومی ربات
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS bot_settings (
            setting_key VARCHAR(50) PRIMARY KEY,
            setting_value VARCHAR(255)
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // درج مقادیر پیش‌فرض برای تایید شماره موبایل (اگر وجود نداشته باشند)
        // phone_verification_enabled: 0 (غیرفعال) | 1 (فعال)
        // phone_verification_mode: iran (فقط ایران) | all (همه کشورها)
        $this->pdo->exec("INSERT IGNORE INTO bot_settings (setting_key, setting_value) VALUES 
            ('phone_verification_enabled', '0'),
            ('phone_verification_mode', 'iran')
        ");

        // --- جداول مربوط به عضویت اجباری (قابلیت جدید) ---

        // جدول لیست کانال‌ها و گروه‌های قفل شده
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS force_join_channels (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chat_id VARCHAR(100), -- آیدی عددی (مثل -100...) یا یوزرنیم (مثل @channel)
            title VARCHAR(255),   -- نام نمایشی برای کاربر
            invite_link VARCHAR(255), -- لینک عضویت (مخصوصا برای کانال‌های خصوصی مهم است)
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

        // درج مقدار پیش‌فرض برای فعال/غیرفعال سازی عضویت اجباری
        // force_join_enabled: 0 (غیرفعال) | 1 (فعال)
        $this->pdo->exec("INSERT IGNORE INTO bot_settings (setting_key, setting_value) VALUES
            ('force_join_enabled', '0')
        ");



    }
    
    
        // --- متد دریافت آمار پیشرفته ربات ---
    public function getAdvancedStats() {
        $stats = [];

        // 1. آمار کاربران
        // تعداد کل
        $stats['users_total'] = $this->pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
        
        // آنلاین (فعالیت در 10 دقیقه گذشته)
        $stats['users_online'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE last_activity >= NOW() - INTERVAL 10 MINUTE")->fetchColumn();
        
        // فعال در 24 ساعت گذشته
        $stats['users_24h'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE last_activity >= NOW() - INTERVAL 24 HOUR")->fetchColumn();
        
        // فعال در یک هفته گذشته
        $stats['users_week'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE last_activity >= NOW() - INTERVAL 7 DAY")->fetchColumn();
        
        // فعال در ماه گذشته
        $stats['users_month'] = $this->pdo->query("SELECT COUNT(*) FROM users WHERE last_activity >= NOW() - INTERVAL 30 DAY")->fetchColumn();

        // 2. آمار تیکت‌ها
        $stats['tickets_total'] = $this->pdo->query("SELECT COUNT(*) FROM tickets")->fetchColumn();
        $stats['tickets_open'] = $this->pdo->query("SELECT COUNT(*) FROM tickets WHERE status = 'open'")->fetchColumn();
        
        // بررسی اولویت‌ها (با فرض اینکه مقادیر high, medium, low ذخیره می‌شوند)
        // اگر مقادیر فارسی ذخیره می‌شوند (زیاد، متوسط، کم) باید کوئری را تغییر داد. فعلا استاندارد انگلیسی را می‌گیریم.
        $stats['tickets_high'] = $this->pdo->query("SELECT COUNT(*) FROM tickets WHERE priority = 'high'")->fetchColumn();
        $stats['tickets_medium'] = $this->pdo->query("SELECT COUNT(*) FROM tickets WHERE priority = 'medium'")->fetchColumn();
        $stats['tickets_low'] = $this->pdo->query("SELECT COUNT(*) FROM tickets WHERE priority = 'low'")->fetchColumn();

        // 3. آمار دپارتمان‌ها
        $stats['depts_total'] = $this->pdo->query("SELECT COUNT(*) FROM departments")->fetchColumn();
        $stats['depts_parents'] = $this->pdo->query("SELECT COUNT(*) FROM departments WHERE parent_id IS NULL")->fetchColumn();
        
        // تعداد زیرمجموعه برای هر والد
        $stats['depts_tree'] = $this->pdo->query("
            SELECT p.title AS parent_name, COUNT(c.id) AS sub_count
            FROM departments p
            LEFT JOIN departments c ON c.parent_id = p.id
            WHERE p.parent_id IS NULL
            GROUP BY p.id
        ")->fetchAll(PDO::FETCH_ASSOC);

        // 4. آمار پشتیبان‌ها (Agents)
        $stats['agents_total'] = $this->pdo->query("SELECT COUNT(DISTINCT agent_id) FROM dept_agents")->fetchColumn();
        
        // تعداد پشتیبان در هر دپارتمان
        $stats['agents_per_dept'] = $this->pdo->query("
            SELECT d.title AS dept_name, COUNT(da.id) AS agent_count
            FROM departments d
            LEFT JOIN dept_agents da ON da.dept_id = d.id
            GROUP BY d.id
            HAVING agent_count > 0
        ")->fetchAll(PDO::FETCH_ASSOC);

        // 5. آمار محصولات
        $stats['products_total'] = $this->pdo->query("SELECT COUNT(*) FROM products")->fetchColumn();

        return $stats;
    }

    
    
}
?>
